<?php

namespace App\Controllers;

use App\Controllers\BaseController;
use App\Models\Backend\SettingEmailModel;
use App\Models\Backend\SettingModel;
use App\Models\Backend\UserModel;
use App\Models\Backend\ActivityModel;

class CronController extends BaseController
{
    public function __construct()
    {
        $this->userModel = new UserModel();
        $this->activityModel = new ActivityModel();
        $this->settingModel = new SettingModel();
        $this->settingEmailModel = new SettingEmailModel();
        $this->session = session();
    }

    //===========================================================//
    public function run_cron_task()
    {
        // Cron task start
        $content = "▶ Cron Automation Tasks Started (" . date('Y-m-d H:i:s') . ")";
        $this->createExecutedFile($content);
        
        $activity_title = lang('Dashboard.cronTasks');
        $activity_body = lang('Dashboard.cronAutomationTasksStarted');
        $this->saveActivity('fas fa-tasks', 'bg-danger', $activity_title, $activity_body);

        // Check cron task permission
        $this->checkPermission();

        // Delete old session
        $session_days_limit = 5;
        $this->deleteOldSession($session_days_limit * 24 * 60 * 60);

        // Check the expiration date of VIP users
        $vip_affected_count = $this->updateVIPUsers(time());
        $content = date('Y-m-d H:i:s') . " → Count of expiration date of VIP users: ". $vip_affected_count;
        $this->createExecutedFile($content);

        // Close inactive tickets after X days
        // Check if "Support Module" is exist
        $supportModulePath = ROOTPATH . "modules/addons/Support/support_module.yaml";
        $supportModuleEnabled = false;  // Variable to determine the state of the plugin
        if (file_exists($supportModulePath)) {
            $count_close_inactive_ticket = $this->closeInactiveSupportTicket();
            $supportModuleEnabled = true; // If the plugin was available, it will be activated
        }

        // Sending success email
        $subject = lang('Dashboard.cronAutomationTasksCompleted');
        $body = "<b>".lang('Dashboard.allCronAutomationTasksCompletedSuccessfully')."</b><br><ul>";
        $body .= "<li>".lang('Dashboard.theSessionsOfTheLastXDaysHaveBeenCleared', [$session_days_limit])."</li>";
        $body .= "<li>".lang('Dashboard.theNumberOfXVIPUsersExpired', [$vip_affected_count])."</li>";
        // Check if the support module is active before adding the ticket related line
        if ($supportModuleEnabled) {
            $body .= "<li>".lang('Dashboard.xInactiveSupportTicketsHaveBeenClosed', [$count_close_inactive_ticket])."</li>";
        }
        $body .= "</ul>";
        $this->sendingEmail($subject, $body);

        // Cron task end
        $content = "⏹︎ Cron Automation Tasks Completed (" . date('Y-m-d H:i:s') . ")\n";
        $this->createExecutedFile($content);

        $activity_title = lang('Dashboard.cronTasks');
        $activity_body = lang('Dashboard.cronAutomationTasksCompleted');
        $this->saveActivity('fas fa-tasks', 'bg-success', $activity_title, $activity_body);
        
        echo lang('Dashboard.cronAutomationTasksCompleted');
        return $this->response->setStatusCode(200, lang('Dashboard.cronAutomationTasksCompleted'));
    }


    //===========================================================//
    private function updateVIPUsers($currentTimestamp)
    {
        $count = $this->userModel->where('vip_role_expiry_at IS NOT NULL')
                      ->where('vip_role_expiry_at <>', 0)
                      ->where('role_id', 11)
                      ->where('vip_role_expiry_at <', $currentTimestamp)
                      ->countAllResults();

        if ($count > 0) {
            $this->userModel->set('role_id', 10)
                            ->set('vip_role_expiry_at', NULL)
                            ->where('vip_role_expiry_at IS NOT NULL')
                            ->where('vip_role_expiry_at <>', 0)
                            ->where('role_id', 11)
                            ->where('vip_role_expiry_at <', $currentTimestamp)
                            ->update();
        }

        $activity_title = lang('Dashboard.cronTasks');
        $activity_body = lang('Dashboard.theNumberOfXVIPUsersExpired', [$count]);
        $this->saveActivity('fas fa-tasks', 'bg-success', $activity_title, $activity_body);
        
        return $count;
    }


    //===========================================================//
    private function updateVIPUsers_SQL($currentTimestamp)
    {
        $db = \Config\Database::connect();

        $sql = "
            UPDATE users_tbl u
            SET u.role_id = 10,
                u.vip_role_expiry_at = NULL
            WHERE u.vip_role_expiry_at IS NOT NULL
            AND u.vip_role_expiry_at <> 0
            AND u.vip_role_expiry_at < ?";
        
        $db->query($sql, [$currentTimestamp]);
        echo time();
    }


    //===========================================================//
    private function deleteOldSession($session_days_limit)
    {
        $dir = WRITEPATH.'session/';
        foreach (glob($dir."ci*") as $file) {
            if (filemtime($file) < time() - $session_days_limit) {
                unlink($file);
            }
        }
        $content = date('Y-m-d H:i:s') . " → Delete old session after ". ($session_days_limit / 24 / 60 / 60) . " day(s).";
        $this->createExecutedFile($content);

        $activity_title = lang('Dashboard.cronTasks');
        $activity_body = lang('Dashboard.theSessionsOfTheLastXDaysHaveBeenCleared', [$session_days_limit / 24 / 60 / 60]);
        $this->saveActivity('fas fa-tasks', 'bg-success', $activity_title, $activity_body);
    }


    //===========================================================//
    private function checkPermission()
    {
        // Retrieve the hostname and resolve IP addresses
        $hostname = gethostname();
        $ip_address1 = gethostbyname($hostname);
        $ip_address2 = "::1"; // IPv6 loopback address
        $ip_address3 = "127.0.0.1"; // IPv4 loopback address

        // List of allowed IP addresses
        $allowedIPs = [
            $ip_address1,
            $ip_address2,
            $ip_address3,
            '111.222.333.444', // Replace with your actual IP addresses
        ];

        // Get the client's IP address
        $clientIP = $this->request->getIPAddress();

        // Check if the client's IP address is in the whitelist
        if (!in_array($clientIP, $allowedIPs)) {
            $content = date('Y-m-d H:i:s') . " → The client's IP address is in the whitelist.\n";
            $this->createExecutedFile($content);
            die('403 - Forbidden IP !');
            return $this->response->setStatusCode(403, 'Forbidden');
        }

        // Environment Check: Ensure the script only runs in the production environment
        if (ENVIRONMENT !== 'production') {
            $content = date('Y-m-d H:i:s') . " → The environment is not production.\n";
            $this->createExecutedFile($content);
            die('403 - Forbidden ENV !');
            return $this->response->setStatusCode(403, 'Forbidden');
        }

        // Security: Ensure this method is not accessible through a browser
        /*if (!$this->request->isCLI()) {
            $content = date('Y-m-d H:i:s') . " → The method is not accessible through a browser.\n";
            $this->createExecutedFile($content);
            die('403 - Forbidden Browser !');
            return $this->response->setStatusCode(403, 'Forbidden');
        }*/

        // Check if the request is coming from curl
        $userAgent = $this->request->getHeaderLine('User-Agent');
        if (stripos($userAgent, 'curl') === false) {
            // The request is not coming from curl
            $content = date('Y-m-d H:i:s') . " → The request is not coming from curl.\n";
            $this->createExecutedFile($content);
            die('403 - Forbidden CURL !');
            return $this->response->setStatusCode(403, 'Forbidden');
        }

        // Check if the request is coming from baseURL in .env
        // Retrieve base URL from .env or config
        $appConfig = config('App'); // Get the App configuration
        $baseURL = rtrim($appConfig->baseURL, '/');

        // Get the current request domain and port
        $requestDomain = $this->request->getServer('HTTP_HOST');
        // Extract domain and port from baseURL (strip http://, https://, and www.)
        $baseDomain = parse_url($baseURL, PHP_URL_HOST);
        $basePort = parse_url($baseURL, PHP_URL_PORT);
        
        // If the port is not defined in the baseURL, assume it's the default for the protocol
        $basePort = $basePort ?? ($this->request->getServer('HTTPS') ? 443 : 80);
        
        // Normalize domains (remove 'www.' if present)
        $normalizedBaseDomain = preg_replace('/^www\./', '', $baseDomain);
        $normalizedRequestDomain = preg_replace('/^www\./', '', $requestDomain);

        // Append port to the base domain for comparison
        if ($basePort && $basePort !== 80 && $basePort !== 443) {
            $normalizedBaseDomain .= ':' . $basePort;
        }

        // Log baseURL and requestDomain for debugging
        $content = date('Y-m-d H:i:s') . " → Base URL: " . $normalizedBaseDomain . ", Request Domain: " . $normalizedRequestDomain . "\n";
        $this->createExecutedFile($content);

        // Check if the request domain matches the base URL
        if ($normalizedBaseDomain !== $normalizedRequestDomain) {
            $content = date('Y-m-d H:i:s') . " → The request domain does not match the base URL.\n";
            $this->createExecutedFile($content);
            die('403 - Forbidden: Invalid domain !');
            return $this->response->setStatusCode(403, 'Forbidden');
        }
    }


    //===========================================================//
    private function sendingEmail($subject, $body)
    {
        $settings = $this->settingModel->select('settings_tbl.email_address, settings_tbl.app_name')->where('settings_tbl.locale', $this->session->lang)->first();
        $to = $settings->email_address;
        $app_name = $settings->app_name;

        $settingsEmail = $this->settingEmailModel->select('settings_email_tbl.*')->where('settings_email_tbl.id', 1)->first();

        helper('email');
        if(sending_cron_email($settingsEmail, $to, $app_name . " - " .$subject, $body)) {
            $content = date('Y-m-d H:i:s') . " → Email sent.";
            $this->createExecutedFile($content);
        }
        
    }


    //===========================================================//
    private function closeInactiveSupportTicket()
    {
        $close_ticket_days = 3;
        $db = db_connect();
        $time_limit = time() - ($close_ticket_days * 24 * 60 * 60);  // محاسبه زمان محدودیت
        
        // Update inactive tickets
        $db->table('mod_support_tickets_tbl')
            ->whereIn('status', ['Open', 'Answered', 'User_Reply'])  // شرط وضعیت‌ها
            ->where('last_reply <=', $time_limit)  // شرط آخرین پاسخ
            ->update(['status' => 'Closed']);  // به‌روزرسانی وضعیت به "بسته‌شده"
        
        // Check the number of updated rows
        $inactive_closed_tickets = $db->affectedRows();

        $db->close();

        $content = date('Y-m-d H:i:s') . " → $inactive_closed_tickets inactive support ticket(s) have been closed. ";
        $this->createExecutedFile($content);

        $activity_title = lang('Dashboard.cronTasks');
        $activity_body = lang('Dashboard.xInactiveSupportTicketsHaveBeenClosed', [$inactive_closed_tickets]);
        $this->saveActivity('fas fa-tasks', 'bg-success', $activity_title, $activity_body);

        return $inactive_closed_tickets;
    }


    //===========================================================//
    private function saveActivity($activity_icon, $activity_background, $activity_title, $activity_body)
    {
        $activity = [
            'user_id'       => 0,
            'icon'          => $activity_icon,
            'background'    => $activity_background,
            'title'         => $activity_title,
            'body'          => $activity_body,
            'ip'            => $_SERVER['REMOTE_ADDR'],
            'agent'         => $_SERVER['HTTP_USER_AGENT'],
            'time'          => time(),
        ];
        $this->activityModel->save($activity);
    }


    //===========================================================//
    private function createExecutedFile($content)
    {
        // Path to the file
        $file_path = WRITEPATH . '/logs/log-cron-tasks.txt';
        
        // Number of days before the file should be reset
        $days_limit = 60;

        // Check if the file exists and its last modified time
        if (file_exists($file_path)) {
            $last_modified_time = filemtime($file_path);
            $current_time = time();
            $time_difference = $current_time - $last_modified_time;
            
            // If the file is older than the limit, delete it
            if ($time_difference > ($days_limit * 24 * 60 * 60)) {
                unlink($file_path); // Delete the file
            }
        }

        // Append new content to the end of the file (or create new file if it doesn't exist)
        if (file_put_contents($file_path, $content . PHP_EOL, FILE_APPEND) === false) {
            return $this->response->setStatusCode(500, 'Failed to write to file.');
        }

        return $this->response->setStatusCode(200, 'Content appended successfully.');
    }

}
