<?php

namespace App\Controllers\Backend;

use App\Controllers\BaseController;
use App\Models\Backend\PageModel;

class PageController extends BaseController
{
    public function __construct()
    {
        $this->session = session();
        $this->pageModel = new PageModel();
    }


    //===========================================================//
    public function add_page()
    {
        $data['pageTitle'] = lang('Dashboard.addPage');
        $data['pageSubTitle'] = lang('Dashboard.addNewPage');
        return view("\App\Views\backend\pages\dynamic\add_page_view", $data);
    }


    //===========================================================//
    public function perform_add_page()
    {
        // Validation the form
        $rules = [
            'title'             => 'required|min_length[2]|max_length[100]',
            'page_contents'     => 'required|min_length[2]',
            'description'       => 'max_length[255]',
            'keywords'          => 'max_length[255]',
            'tags'              => 'max_length[255]',
            'type'              => 'integer|required',
            'status'            => 'permit_empty',
            'breadcrumbs'       => [
                'ext_in[breadcrumbs,png,jpg,jpeg]',
                'is_image[breadcrumbs]',
                'max_size[breadcrumbs,2048]',
                'errors' => [
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 2048 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url("dashboard/add-page"))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $newName = "breadcrumbs.png";
        $img = $this->request->getFile('breadcrumbs');
        if($img != "") {
            $newName = $img->getRandomName();
            $imageExtension = $img->getExtension();
            $img->move('./backend/assets/uploads/admins/page/breadcrumbs/', $newName);

            $imgPath = $img->getTempName();
            $imgQuality = 75;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#000000',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'left', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'bottom', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 30,
                    'fontPath'   => ''
                ])
                ->fit(1920, 440, 'center')
                ->save('./backend/assets/uploads/admins/page/breadcrumbs/'.$newName, $imgQuality);
        }


        $status = 0;
        if($this->request->getVar('status') == 'on') $status = 1;

        $pageData = [
            'title'             => $this->request->getVar('title'),
            'title_slug'        => url_title($this->request->getVar('title'), '-', true),
            'page_contents'     => htmlspecialchars($this->request->getVar('page_contents')),
            'description'       => $this->request->getVar('description'),
            'keywords'          => $this->request->getVar('keywords'),
            'tags'              => $this->request->getVar('tags'),
            'tags_slug'         => url_title($this->request->getVar('tags'),'-',true),
            'type'              => $this->request->getVar('type'),
            'status'            => $status,
            'locale'            => $this->session->lang,
            'breadcrumbs'       => $newName,
        ];
        $this->pageModel->save($pageData);

        return redirect()->to(base_url("dashboard/pages-list"))->with('success', lang('Dashboard.postAddedSuccessfully'));
    }



    //===========================================================//
    public function pages_list()
    {
        $locale = $this->session->lang;
        $keyword = "";
        if(isset($_GET['keyword'])) $keyword = $_GET['keyword'];

        if(empty($keyword))
        {
            $data = [
                'pages' => $this->pageModel
                    ->select('pages_tbl.*')
                    ->Where('pages_tbl.locale', $locale)
                    ->Where('pages_tbl.deleted_at', NULL)
                    ->orderBy('pages_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->pageModel->pager
            ];

        }else{
            $data = [
                'pages' => $this->pageModel
                    ->select('pages_tbl.*')
                    ->Where('pages_tbl.locale', $locale)
                    ->Where('pages_tbl.deleted_at', NULL) // To prevent show softDeleted row
                    ->like('pages_tbl.title',$keyword)
                    ->Where('pages_tbl.deleted_at', NULL) // To prevent show softDeleted row
                    ->orderBy('pages_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->pageModel->pager
            ];
        }

        $data['pageTitle'] = lang('Dashboard.pagesList');
        $data['pageSubTitle'] = lang('Dashboard.pagesList');
        return view('\App\Views\backend\pages\dynamic\pages_list_view', $data);
    }


    //===========================================================//
    public function edit_page()
    {
        // Check post exist
        $id = service('uri')->getSegment(3);
        $data['page'] = $this->pageModel->where('id', $id)->withDeleted()->first();
        if(is_null($data['page']))
        {
            return redirect()->to(base_url('dashboard/pages-list'))->with('error', lang('Dashboard.notFound'));
        }

        $data['pageTitle'] = lang('Dashboard.editPage');
        $data['pageSubTitle'] = lang('Dashboard.editPage');
        return view('\App\Views\backend\pages\dynamic\edit_page_view', $data);
    }


    //===========================================================//
    public function perform_edit_page()
    {
        // Validation the form
        $rules = [
            'id'                => 'required|integer',
            'title'             => 'required|min_length[2]|max_length[100]',
            'page_contents'     => 'required|min_length[2]',
            'description'       => 'max_length[255]',
            'keywords'          => 'max_length[255]',
            'tags'              => 'max_length[255]',
            'type'              => 'integer|required',
            'status'            => 'permit_empty',
            'old_breadcrumbs'   => 'required',
            'breadcrumbs'       => [
                'ext_in[breadcrumbs,png,jpg,jpeg]',
                'is_image[breadcrumbs]',
                'max_size[breadcrumbs,2048]',
                'errors' => [
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 2048 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ];

        $id = $this->request->getVar('id');
        $old_breadcrumbs = $this->request->getVar('old_breadcrumbs');

        if (!$this->validate($rules)) {
            return redirect()->to(base_url("dashboard/edit-page/$id"))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $newName = $old_breadcrumbs;
        $img = $this->request->getFile('breadcrumbs');
        if($img != "") {
            $newName = $img->getRandomName();
            $imageExtension = $img->getExtension();
            $img->move('./backend/assets/uploads/admins/page/breadcrumbs/', $newName);

            $imgPath = $img->getTempName();
            $imgQuality = 75;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#000000',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'left', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'bottom', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 30,
                    'fontPath'   => ''
                ])
                ->fit(1920, 440, 'center')
                ->save('./backend/assets/uploads/admins/page/breadcrumbs/'.$newName, $imgQuality);

            // Remove old_image from disk
            if($old_breadcrumbs != 'breadcrumbs.png')
            {
                @unlink('./backend/assets/uploads/admins/page/breadcrumbs/'.$old_breadcrumbs);
            }
        }

        $status = 0;
        if($this->request->getVar('status') == 'on') $status = 1;

        $pageData = [
            'id'                => $this->request->getVar('id'),
            'title'             => $this->request->getVar('title'),
            'title_slug'        => $this->request->getVar('title_slug'),
            'page_contents'     => htmlspecialchars($this->request->getVar('page_contents')),
            'description'       => $this->request->getVar('description'),
            'keywords'          => $this->request->getVar('keywords'),
            'tags'              => $this->request->getVar('tags'),
            'tags_slug'         => url_title($this->request->getVar('tags'),'-',true),
            'type'              => $this->request->getVar('type'),
            'status'            => $status,
            'locale'            => $this->session->lang,
            'breadcrumbs'       => $newName,
        ];
        $this->pageModel->save($pageData);

        return redirect()->to(base_url("dashboard/pages-list"))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function delete_page($id = NULL)
    {
        // Check the page is exist
        $id = service('uri')->getSegment(3);
        $data['page'] = $this->pageModel->where('id', $id)->first();
        if(is_null($data['page']))
            return redirect()->to(base_url('dashboard/pages-list'))->with('error', lang('Dashboard.notFound'));

        // softDelete post
        $this->pageModel->delete($id);

        return redirect()->to(base_url('dashboard/pages-list'))->with('success', lang('Dashboard.dataDeletedSuccessfully'));
    }
}
