<?php

namespace App\Controllers\Backend;

use App\Controllers\BaseController;
use App\Models\Backend\CategoryModel;
use App\Models\Backend\BlogModel;

class CategoryController extends BaseController
{
    public function __construct()
    {
        $this->categoryModel = new CategoryModel();
        $this->blogModel = new BlogModel();
        $this->session = session();
    }


    //===========================================================//
	public function create_category()
	{
        $locale = $this->session->lang;
        $db = db_connect();

        // Fetch user roles
        $builder = $db->table('roles_tbl');
        $builder->where('active', 1);
        $builder->orderBy('id', 'ASC');
        $data['roles'] = $builder->get()->getResult();

        // Fetch category types
        $builder = $db->table('categories_types_tbl');
        $builder->where('status', 1);
        $builder->orderBy('id', 'ASC');
        $data['categoriesTypes'] = $builder->get()->getResult();

        // Fetch main categories
        $builder = $db->table('categories_tbl');
        $builder->where('locale', $locale);
        $builder->where('deleted_at', NULL);
        $builder->where('status', 1);
        $builder->where('parent_id', 0);
        $builder->orderBy('id', 'ASC');
        $data['mainCategories'] = $builder->get()->getResult();

        $db->close();

        $data['pageTitle'] = lang('Dashboard.createCategory');
        $data['pageSubTitle'] = lang('Dashboard.createCategory');
        return view('\App\Views\backend\categories\create_category_view', $data);
	}


    //===========================================================//
    public function perform_create_category()
    {
        // Validation the form
        $rules = [
            'title'         => 'required|min_length[2]|max_length[60]',
            'icon'          => 'permit_empty|max_length[100]',
            'color'         => 'permit_empty|max_length[20]',
            'type_id'       => 'required',
            'parent_id'     => 'required',
            'order'         => 'required',
            'image'         => [
                'ext_in[image,png,jpg,jpeg]',
                'is_image[image]',
                'max_size[image,200]',
                'errors' => [
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 200 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url("dashboard/create-category"))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $newName = "category.png";
        $img = $this->request->getFile('image');
        if($img != "") {
            $newName = $img->getRandomName();
            $imageExtension = $img->getExtension();
            $img->move('./backend/assets/uploads/admins/categories/', $newName);

            $imgPath = $img->getTempName();
            $imgQuality = 90;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(256, 256, 'center')
                ->save('./backend/assets/uploads/admins/categories/'.$newName, $imgQuality);
        }

        $status = 0;
        if($this->request->getVar('status') == "on") $status = 1;

        $categoryData = [
            'title'         => $this->request->getVar('title'),
            'parent_id'     => $this->request->getVar('parent_id'),
            'title_slug'    => url_title($this->request->getVar('title'), '-', true),
            'icon'          => $this->request->getVar('icon'),
            'color'         => $this->request->getVar('color'),
            'type_id'       => $this->request->getVar('type_id'),
            'role_id'       => 1,
            'order'         => $this->request->getVar('order'),
            'locale'        => $this->session->lang,
            'image'         => $newName,
            'status'        => $status,
        ];
        $this->categoryModel->save($categoryData);

        return redirect()->to(base_url("dashboard/create-category"))->with('success', lang('Dashboard.categoryCreatedSuccessfully'));
    }


    //===========================================================//
    public function categories_list()
    {
        $locale = $this->session->lang;
        $keyword = "";
        if(isset($_GET['keyword'])) $keyword = $_GET['keyword'];

        if(empty($keyword))
        {
            $data = [
                'categories' => $this->categoryModel
                    ->select('categories_tbl.*, categories_types_tbl.title AS category_type_title')
                    ->Where('categories_tbl.locale', $locale)
                    ->Where('categories_tbl.deleted_at', NULL)
                    ->join('categories_types_tbl', 'categories_types_tbl.id = categories_tbl.type_id')
                    ->orderBy('categories_tbl.order', 'ASC')
                    ->paginate(10),
                'pager' => $this->categoryModel->pager
            ];

        }else{
            $data = [
                'categories' => $this->categoryModel
                    ->select('categories_tbl.*, categories_types_tbl.title AS category_type_title')
                    ->like('categories_tbl.title',$keyword)
                    ->orLike('categories_types_tbl.title',$keyword)
                    ->Where('categories_tbl.locale', $locale)
                    ->Where('categories_tbl.deleted_at', NULL) // To prevent show softDeleted row
                    ->join('categories_types_tbl', 'categories_types_tbl.id = categories_tbl.type_id')
                    ->orderBy('categories_tbl.order', 'ASC')
                    ->paginate(10),
                'pager' => $this->categoryModel->pager
            ];
        }

        $data['pageTitle'] = lang('Dashboard.categoriesList');
        $data['pageSubTitle'] = lang('Dashboard.categoriesList');
        return view('\App\Views\backend\categories\categories_list_view', $data);
    }


    //===========================================================//
    public function delete_category()
    {
        // Check category exist
        $category_id = service('uri')->getSegment(3);
        $category = $this->categoryModel->where('id', $category_id)->first();
        if(is_null($category))
        {
            return redirect()->to(base_url('dashboard/categories-list'))->with('error', lang('Dashboard.notFound'));

        }else{
            // Check if there are sub category in this category
            $subCategory = $this->categoryModel->where('parent_id', $category_id)->first();
            if(! is_null($subCategory))
            {
                return redirect()->to(base_url('dashboard/categories-list'))->with('error', lang('Dashboard.thereAreSubCategoryInThisCategoryFirstDeleteSubCategory'));
            }

            // Check if there are content in this category
            // ...

            // Check if there are posts in this category
            $post = $this->blogModel->where('category_id', $category_id)->first();
            if(! is_null($post))
            {
                return redirect()->to(base_url('dashboard/categories-list'))->with('error', lang('Dashboard.thisCategoryHasPosted.firstDeleteOrMovePostsToAnotherCategory'));
            }

            // Delete category
            $this->categoryModel->delete($category_id);

            // Remove image from disk
            if($category->image != 'category.png')
                @unlink('./backend/assets/uploads/admins/categories/'.$category->image);

            return redirect()->to(base_url('dashboard/categories-list'))->with('success', lang('Dashboard.dataDeletedSuccessfully'));
        }
    }


    //===========================================================//
    public function edit_category($id = NULL)
    {
        $category_id = service('uri')->getSegment(3);
        $locale = $this->session->lang;

        // Check category exist or not
        $data['category'] = $this->categoryModel->where('id', $category_id)->first();
        if(is_null($data['category']))
            return redirect()->back()->with('error', lang('Dashboard.notFound'));

        $db = db_connect();

        // Fetch user roles
        $builder = $db->table('roles_tbl');
        $builder->where('active', 1);
        $builder->orderBy('id', 'ASC');
        $data['roles'] = $builder->get()->getResult();

        // Fetch category types
        $builder = $db->table('categories_types_tbl');
        $builder->where('status', 1);
        $builder->orderBy('id', 'ASC');
        $data['categoriesTypes'] = $builder->get()->getResult();

        // Fetch main categories
        $builder = $db->table('categories_tbl');
        $builder->where('status', 1);
        $builder->where('parent_id', 0);
        $builder->where('locale', $locale);
        $builder->where('deleted_at', NULL);
        $builder->orderBy('id', 'ASC');
        $data['mainCategories'] = $builder->get()->getResult();

        $db->close();

        $data['category'] = $this->categoryModel->select('categories_tbl.*')
            ->where('categories_tbl.id', $category_id)
            ->first();

        $data['pageTitle'] = lang('Dashboard.editCategory');
        $data['pageSubTitle'] = lang('Dashboard.editCategory');
        return view('\App\Views\backend\categories\edit_category_view', $data);
    }


    //===========================================================//
    public function perform_edit_category()
    {
        $id = $this->request->getVar('id');
        $old_image = $this->request->getVar('old_image');
        // Validation the form
        $rules = [
            'id'            => 'required|integer',
            'old_image'     => 'required',
            'title'         => 'required|min_length[2]|max_length[60]',
            'icon'          => 'permit_empty|max_length[60]',
            'color'         => 'permit_empty|max_length[20]',
            'type_id'       => 'required',
            'parent_id'     => 'required',
            'order'         => 'required',
            'image'         => [
                'ext_in[image,png,jpg,jpeg]',
                'is_image[image]',
                'max_size[image,200]',
                'errors' => [
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 200 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url("dashboard/edit-category/$id"))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $newName = "category.png";
        $img = $this->request->getFile('image');
        if($img != "") {
            $newName = $img->getRandomName();
            $imageExtension = $img->getExtension();
            $img->move('./backend/assets/uploads/admins/categories/', $newName);

            $imgPath = $img->getTempName();
            $imgQuality = 90;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(256, 256, 'center')
                ->save('./backend/assets/uploads/admins/categories/'.$newName, $imgQuality);
        }


        $status = 0;
        if($this->request->getVar('status') == "on") $status = 1;

        if($img != "")
        {
            // Update category with image and delete old image
            $data = [
                'id' => $this->request->getVar('id'),
                'title' => $this->request->getVar('title'),
                'title_slug' => url_title($this->request->getVar('title'), '-', true),
                'icon' => $this->request->getVar('icon'),
                'color' => $this->request->getVar('color'),
                'type_id' => $this->request->getVar('type_id'),
                'parent_id' => $this->request->getVar('parent_id'),
                'order' => $this->request->getVar('order'),
                'locale' => $this->session->lang,
                'status' => $status,
                'image' => $newName,
            ];
            $this->categoryModel->save($data);

            // Remove old image from disk
            if($old_image != 'category.png')
                @unlink('./backend/assets/uploads/admins/categories/'.$old_image);

        }else{
            // Update category without image
            $data = [
                'id' => $this->request->getVar('id'),
                'title' => $this->request->getVar('title'),
                'title_slug' => url_title($this->request->getVar('title'), '-', true),
                'icon' => $this->request->getVar('icon'),
                'color' => $this->request->getVar('color'),
                'type_id' => $this->request->getVar('type_id'),
                'parent_id' => $this->request->getVar('parent_id'),
                'order' => $this->request->getVar('order'),
                'locale' => $this->session->lang,
                'status' => $status,
            ];
            $this->categoryModel->save($data);
        }

        return redirect()->to(base_url("dashboard/categories-list"))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }
}
