<?php

namespace App\Controllers\Backend;

use App\Controllers\BaseController;
use App\Models\Backend\CategoryModel;
use App\Models\Backend\BlogModel;
use App\Models\Backend\BlogCommentModel;

class BlogController extends BaseController
{
    public function __construct()
    {
        $this->categoryModel = new CategoryModel();
        $this->blogModel = new BlogModel();
        $this->blogCommentModel = new BlogCommentModel();
        $this->session = session();
    }


    //===========================================================//
    public function add_post()
    {
        $locale = $this->session->lang;
        $db = db_connect();

        // Fetch structures
        $builder = $db->table('blog_structure_tbl');
        $builder->where('status', 1);
        $builder->orderBy('id', 'ASC');
        $data['blogStructure'] = $builder->get()->getResult();

        // Fetch main categories
        $builder = $db->table('categories_tbl');
        $builder->where('locale', $locale);
        $builder->where('status', 1);
        $builder->where('parent_id', 0);
        $builder->where('type_id', 3);
        $builder->where('deleted_at', NULL);
        $builder->orderBy('id', 'ASC');
        $data['blogMainCategories'] = $builder->get()->getResult();

        $db->close();

        $data['pageTitle'] = lang('Dashboard.addPost');
        $data['pageSubTitle'] = lang('Dashboard.addNewPost');
        return view("\App\Views\backend\blog\add_post_view", $data);
    }


    //===========================================================//
    public function perform_add_post()
    {
        // Validation the form
        $rules = [
            'title'             => 'required|min_length[2]|max_length[100]',
            'post_contents'     => 'required|min_length[2]',
            'description'       => 'max_length[255]',
            'keywords'          => 'max_length[255]',
            'tags'              => 'max_length[255]',
            'reference_title'   => 'max_length[60]',
            'reference_url'     => 'max_length[255]',
            'structure_id'      => 'required',
            'category_id'       => 'required',
            'is_advertising'    => 'permit_empty',
            'allow_comments'    => 'permit_empty',
            'special'           => 'permit_empty',
            'featured'          => 'permit_empty',
            'status'            => 'permit_empty',
            'image'         => [
                'ext_in[image,png,jpg,jpeg]',
                'is_image[image]',
                'max_size[image,1024]',
                'errors' => [
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 1024 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url("dashboard/add-post"))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $newName = "post.png";
        $img = $this->request->getFile('image');
        if($img != "") {
            $newName = $img->getRandomName();
            $imageExtension = $img->getExtension();
            $img->move('./backend/assets/uploads/admins/blog/', $newName);

            $imgPath = $img->getTempName();
            $imgQuality = 75;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#000000',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'left', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'bottom', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 30,
                    'fontPath'   => ''
                ])
                ->fit(850, 350, 'center')
                ->save('./backend/assets/uploads/admins/blog/'.$newName, $imgQuality);

            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#000000',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'left', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'bottom', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 30,
                    'fontPath'   => ''
                ])
                ->fit(350, 150, 'center')
                ->save('./backend/assets/uploads/admins/blog/thumb/'.$newName, $imgQuality);
        }


        $is_advertising = $allow_comments = $special = $featured = $status = 0;
        if($this->request->getVar('is_advertising') == 'on') $is_advertising = 1;
        if($this->request->getVar('allow_comments') == 'on') $allow_comments = 1;
        if($this->request->getVar('special') == 'on') $special = 1;
        if($this->request->getVar('featured') == 'on') $featured = 1;
        if($this->request->getVar('status') == 'on') $status = 1;

        $postData = [
            'user_id'           => $this->session->id,
            'title'             => $this->request->getVar('title'),
            'title_slug'        => url_title($this->request->getVar('title'), '-', true),
            'post_contents'     => htmlspecialchars($this->request->getVar('post_contents')),
            'description'       => $this->request->getVar('description'),
            'keywords'          => $this->request->getVar('keywords'),
            'tags'              => $this->request->getVar('tags'),
            'tags_slug'         => url_title($this->request->getVar('tags'),'-',true),
            'reference_title'   => $this->request->getVar('reference_title'),
            'reference_url'     => $this->request->getVar('reference_url'),
            'structure_id'      => $this->request->getVar('structure_id'),
            'url'               => $this->request->getVar('url'),
            'category_id'       => $this->request->getVar('category_id'),
            'is_advertising'    => $is_advertising,
            'allow_comments'    => $allow_comments,
            'special'           => $special,
            'featured'          => $featured,
            'status'            => $status,
            'locale'            => $this->session->lang,
            'image'             => $newName,
        ];
        $this->blogModel->save($postData);

        return redirect()->to(base_url("dashboard/posts-list"))->with('success', lang('Dashboard.postAddedSuccessfully'));
    }


    //===========================================================//
    public function posts_list()
    {
        $locale = $this->session->lang;
        $keyword = "";
        if(isset($_GET['keyword'])) $keyword = $_GET['keyword'];

        if(empty($keyword))
        {
            $data = [
                'posts' => $this->blogModel
                    ->select('blog_tbl.*, categories_tbl.title AS category_title, users_tbl.first_name, users_tbl.last_name')
                    ->Where('blog_tbl.locale', $locale)
                    ->Where('blog_tbl.deleted_at', NULL)
                    ->join('categories_tbl', 'categories_tbl.id = blog_tbl.category_id')
                    ->join('users_tbl', 'users_tbl.id = blog_tbl.user_id')
                    ->orderBy('blog_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->blogModel->pager
            ];

        }else{
            $data = [
                'posts' => $this->blogModel
                    ->select('blog_tbl.*, categories_tbl.title AS category_title, users_tbl.first_name, users_tbl.last_name')
                    ->Where('blog_tbl.locale', $locale)
                    ->Where('blog_tbl.deleted_at', NULL) // To prevent show softDeleted row
                    ->like('blog_tbl.title',$keyword)
                    ->Where('blog_tbl.deleted_at', NULL) // To prevent show softDeleted row
                    ->join('categories_tbl', 'categories_tbl.id = blog_tbl.category_id')
                    ->join('users_tbl', 'users_tbl.id = blog_tbl.user_id')
                    ->orderBy('blog_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->blogModel->pager
            ];
        }

        $data['pageTitle'] = lang('Dashboard.postsList');
        $data['pageSubTitle'] = lang('Dashboard.postsList');
        return view('\App\Views\backend\blog\posts_list_view', $data);
    }


    //===========================================================//
    public function soft_deleted_posts_list()
    {
        $locale = $this->session->lang;
        $keyword = "";
        if(isset($_GET['keyword'])) $keyword = $_GET['keyword'];

        if(empty($keyword))
        {
            $data = [
                'posts' => $this->blogModel
                    ->select('blog_tbl.*, categories_tbl.title AS category_title, users_tbl.first_name, users_tbl.last_name')
                    ->Where('blog_tbl.locale', $locale)
                    ->onlyDeleted()
                    ->join('categories_tbl', 'categories_tbl.id = blog_tbl.category_id')
                    ->join('users_tbl', 'users_tbl.id = blog_tbl.user_id')
                    ->orderBy('blog_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->blogModel->pager
            ];

        }else{
            $data = [
                'posts' => $this->blogModel
                    ->select('blog_tbl.*, categories_tbl.title AS category_title, users_tbl.first_name, users_tbl.last_name')
                    ->Where('blog_tbl.locale', $locale)
                    ->onlyDeleted()
                    ->like('blog_tbl.title',$keyword)
                    ->onlyDeleted()
                    ->join('categories_tbl', 'categories_tbl.id = blog_tbl.category_id')
                    ->join('users_tbl', 'users_tbl.id = blog_tbl.user_id')
                    ->orderBy('blog_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->blogModel->pager
            ];
        }

        $data['pageTitle'] = lang('Dashboard.deletedPostsList');
        $data['pageSubTitle'] = lang('Dashboard.deletedPostsList');
        return view('\App\Views\backend\blog\soft_deleted_posts_list_view', $data);
    }


    //===========================================================//
    public function edit_post()
    {
        // Check post exist
        $post_id = service('uri')->getSegment(3);
        $data['post'] = $this->blogModel->where('id', $post_id)->withDeleted()->first();
        if(is_null($data['post']))
        {
            return redirect()->to(base_url('dashboard/posts-list'))->with('error', lang('Dashboard.notFound'));
        }

        $locale = $this->session->lang;
        $db = db_connect();

        // Fetch structures
        $builder = $db->table('blog_structure_tbl');
        $builder->where('status', 1);
        $builder->orderBy('id', 'ASC');
        $data['blogStructure'] = $builder->get()->getResult();

        // Fetch main categories
        $builder = $db->table('categories_tbl');
        $builder->where('locale', $locale);
        $builder->where('status', 1);
        $builder->where('parent_id', 0);
        $builder->where('type_id', 3);
        $builder->orderBy('id', 'ASC');
        $data['blogMainCategories'] = $builder->get()->getResult();

        $db->close();

        $data['pageTitle'] = lang('Dashboard.editPost');
        $data['pageSubTitle'] = lang('Dashboard.editPost');
        return view('\App\Views\backend\blog\edit_post_view', $data);
    }


    //===========================================================//
    public function perform_edit_post()
    {
        $id = $this->request->getVar('id');
        $old_image = $this->request->getVar('old_image');
        // Validation the form
        $rules = [
            'id'                => 'required|integer',
            'title'             => 'required|min_length[2]|max_length[100]',
            'title_slug'        => 'required|min_length[2]|max_length[120]',
            'post_contents'     => 'required|min_length[2]',
            'description'       => 'required|min_length[2]|max_length[255]',
            'keywords'          => 'max_length[255]',
            'tags'              => 'max_length[255]',
            'reference_title'   => 'max_length[60]',
            'reference_url'     => 'max_length[255]',
            'structure_id'      => 'required',
            'category_id'       => 'required',
            'old_image'         => 'required',
            'is_advertising'    => 'permit_empty',
            'allow_comments'    => 'permit_empty',
            'special'           => 'permit_empty',
            'featured'          => 'permit_empty',
            'status'            => 'permit_empty',
            'image'         => [
                'ext_in[image,png,jpg,jpeg]',
                'is_image[image]',
                'max_size[image,1024]',
                'errors' => [
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 1024 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url("dashboard/edit-post/$id"))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $newName = $old_image;
        $img = $this->request->getFile('image');
        if($img != "") {
            $newName = $img->getRandomName();
            $imageExtension = $img->getExtension();
            $img->move('./backend/assets/uploads/admins/blog/', $newName);

            $imgPath = $img->getTempName();
            $imgQuality = 75;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#000000',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'left', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'bottom', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 30,
                    'fontPath'   => ''
                ])
                ->fit(850, 350, 'center')
                ->save('./backend/assets/uploads/admins/blog/'.$newName, $imgQuality);

            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#000000',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'left', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'bottom', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 30,
                    'fontPath'   => ''
                ])
                ->fit(350, 150, 'center')
                ->save('./backend/assets/uploads/admins/blog/thumb/'.$newName, $imgQuality);

            // Remove old_image from disk
            if($old_image != 'post.png')
            {
                @unlink('./backend/assets/uploads/admins/blog/'.$old_image);
                @unlink('./backend/assets/uploads/admins/blog/thumb/'.$old_image);
            }
        }


        $is_advertising = $allow_comments = $special = $featured = $status = 0;
        if($this->request->getVar('is_advertising') == 'on') $is_advertising = 1;
        if($this->request->getVar('allow_comments') == 'on') $allow_comments = 1;
        if($this->request->getVar('special') == 'on') $special = 1;
        if($this->request->getVar('featured') == 'on') $featured = 1;
        if($this->request->getVar('status') == 'on') $status = 1;

        $postData = [
            'id'                => $this->request->getVar('id'),
            'title'             => $this->request->getVar('title'),
            'title_slug'        => $this->request->getVar('title_slug'),
            'post_contents'     => htmlspecialchars($this->request->getVar('post_contents')) ,
            'description'       => $this->request->getVar('description'),
            'keywords'          => $this->request->getVar('keywords'),
            'tags'              => $this->request->getVar('tags'),
            'tags_slug'         => url_title($this->request->getVar('tags'), '-', true),
            'reference_title'   => $this->request->getVar('reference_title'),
            'reference_url'     => $this->request->getVar('reference_url'),
            'structure_id'      => $this->request->getVar('structure_id'),
            'url'               => $this->request->getVar('url'),
            'category_id'       => $this->request->getVar('category_id'),
            'is_advertising'    => $is_advertising,
            'allow_comments'    => $allow_comments,
            'status'            => $status,
            'special'           => $special,
            'featured'          => $featured,
            'locale'            => $this->session->lang,
            'image'             => $newName,
        ];
        $this->blogModel->save($postData);

        return redirect()->to(base_url("dashboard/posts-list"))->with('success', lang('Dashboard.postEditedSuccessfully'));
    }


    //===========================================================//
    public function delete_post($id = NULL)
    {
        // Check the post is exist
        $id = service('uri')->getSegment(3);
        $data['post'] = $this->blogModel->where('id', $id)->first();
        if(is_null($data['post']))
            return redirect()->to(base_url('dashboard/posts-list'))->with('error', lang('Dashboard.notFound'));

        // softDelete post
        $this->blogModel->delete($id);

        return redirect()->to(base_url('dashboard/posts-list'))->with('success', lang('Dashboard.dataDeletedSuccessfully'));
    }


    //===========================================================//
    public function post_restore($id = NULL)
    {
        // Check the post is exist and soft deleted
        $id = service('uri')->getSegment(3);
        $data['post'] = $this->blogModel->where('id', $id)->onlyDeleted()->first();
        if(is_null($data['post']))
            return redirect()->to(base_url('dashboard/soft-deleted-posts-list'))->with('error', lang('Dashboard.notFound'));

        $id = service('uri')->getSegment(3);
        $this->blogModel->set('deleted_at', NULL);
        $this->blogModel->where('id', $id);
        if(! $this->blogModel->update()) {
            return redirect()->to(base_url('dashboard/soft-deleted-posts-list/'))->withInput()->with('errors', $this->blogModel->errors());
        }

        return redirect()->to(base_url('dashboard/soft-deleted-posts-list/'))->withInput()->with('success', lang('Dashboard.postRestoredSuccessfully'));
    }


    //===========================================================//
    public function hard_delete_post($id = NULL)
    {
        // Check the post is exist and soft deleted
        $id = service('uri')->getSegment(3);
        $post = $this->blogModel->where('id', $id)->onlyDeleted()->first();
        if(is_null($post))
            return redirect()->to(base_url('dashboard/soft-deleted-posts-list'))->with('error', lang('Dashboard.notFound'));

        // Hard delete post
        $db = db_connect();
        $builder = $db->table('blog_tbl');
        $builder->delete(['id' => $id]);

        // Remove old_image from disk
        if($post->image != 'post.png')
        {
            @unlink('./backend/assets/uploads/admins/blog/'.$post->image);
            @unlink('./backend/assets/uploads/admins/blog/thumb/'.$post->image);
        }
        $db->close();

        return redirect()->to(base_url('dashboard/soft-deleted-posts-list'))->with('success', lang('Dashboard.postDeletedSuccessfully'));
    }


    //===========================================================//
    public function blog_comments_list()
    {
        $keyword = "";
        if(isset($_GET['keyword'])) $keyword = $_GET['keyword'];

        if(empty($keyword))
        {
            $data = [
                'blogComments' => $this->blogCommentModel
                    ->select('blog_comments_tbl.*')
                    ->orderBy('blog_comments_tbl.approved', 'ASC')
                    ->orderBy('blog_comments_tbl.id', 'DESC')
                    ->paginate(20),
                'pager' => $this->blogCommentModel->pager
            ];

        }else{
            $data = [
                'blogComments' => $this->blogCommentModel
                    ->select('blog_comments_tbl.*')
                    ->like('blog_comments_tbl.author_name', $keyword)
                    ->orLike('blog_comments_tbl.author_email', $keyword)
                    ->orLike('blog_comments_tbl.author_ip', $keyword)
                    ->orLike('blog_comments_tbl.comment', $keyword)
                    ->orderBy('blog_comments_tbl.approved', 'ASC')
                    ->orderBy('blog_comments_tbl.id', 'DESC')
                    ->paginate(20),
                'pager' => $this->blogCommentModel->pager
            ];
        }

        $data['pageTitle'] = lang('Dashboard.usersComments');
        $data['pageSubTitle'] = lang('Dashboard.usersComments');
        return view('\App\Views\backend\blog\blog_comments_list_view', $data);
    }


    //===========================================================//
    public function deleted_blog_comments_list()
    {
        $keyword = "";
        if(isset($_GET['keyword'])) $keyword = $_GET['keyword'];

        if(empty($keyword))
        {
            $data = [
                'blogComments' => $this->blogCommentModel
                    ->select('blog_comments_tbl.*')
                    ->orderBy('blog_comments_tbl.approved', 'ASC')
                    ->orderBy('blog_comments_tbl.id', 'DESC')
                    ->onlyDeleted()
                    ->paginate(20),
                'pager' => $this->blogCommentModel->pager
            ];

        }else{
            $data = [
                'blogComments' => $this->blogCommentModel
                    ->select('blog_comments_tbl.*')
                    ->like('blog_comments_tbl.author_name', $keyword)
                    ->orLike('blog_comments_tbl.author_email', $keyword)
                    ->orLike('blog_comments_tbl.author_ip', $keyword)
                    ->orLike('blog_comments_tbl.comment', $keyword)
                    ->orderBy('blog_comments_tbl.approved', 'ASC')
                    ->orderBy('blog_comments_tbl.id', 'DESC')
                    ->onlyDeleted()
                    ->paginate(20),
                'pager' => $this->blogCommentModel->pager
            ];
        }

        $data['pageTitle'] = lang('Dashboard.deletedUsersComments');
        $data['pageSubTitle'] = lang('Dashboard.deletedUsersComments');
        return view('\App\Views\backend\blog\deleted_blog_comments_list_view', $data);
    }


    //===========================================================//
    public function perform_approve_unapprove_blog_comments()
    {
        $rules = [
            'post_id'  	    => 'required',
            'approved'  	=> 'permit_empty',
        ];

        if (!$this->validate($rules)) {
            $errors = ['errors', $this->validator->getErrors()];
            json_encode($errors);
            echo lang('Frontend.pleaseFillAllRequiredFieldsCorrectly');
            die();
        }

        // Valid submission
        $post_id = $this->request->getPost('post_id');
        $approved = $this->request->getPost('approved'); // 0: No | 1: Yes

        // Check comment exist
        $data['comment'] = $this->blogCommentModel->where('id', $post_id)->withDeleted()->first();
        if(is_null($data['comment']))
        {
            echo lang('Dashboard.notFound');
        }

        if($approved != 1) {
            $newData = [
                'id'           => $post_id,
                'approved'     => 1,
            ];
            $this->blogCommentModel->save($newData);
            echo lang('Dashboard.commentApprovedSuccessfully');

        }else{
            $newData = [
                'id'           => $post_id,
                'approved'     => 0,
            ];
            $this->blogCommentModel->save($newData);
            echo lang('Dashboard.commentUnapprovedSuccessfully');
        }
        
    }


    //===========================================================//
    public function delete_blog_comment()
    {
        // Check review exist
        $post_id = service('uri')->getSegment(3);
        $data['comment'] = $this->blogCommentModel->where('id', $post_id)->withDeleted()->first();
        if(is_null($data['comment']))
        {
            return redirect()->back()->with('error', lang('Dashboard.notFound'));
        }

        $newData = [
            'id'           => $post_id,
            'approved'     => 0,
        ];
        $this->blogCommentModel->save($newData);

        // softDelete
        $this->blogCommentModel->delete($post_id);
        return redirect()->to(base_url('dashboard/blog-comments-list/'))->withInput()->with('success', lang('Dashboard.dataDeletedSuccessfully'));
    }


    //===========================================================//
    public function restore_blog_comment()
    {
        // Check review exist
        $post_id = service('uri')->getSegment(3);
        $data['comment'] = $this->blogCommentModel->where('id', $post_id)->withDeleted()->first();
        if(is_null($data['comment']))
        {
            return redirect()->back()->with('error', lang('Dashboard.notFound'));
        }

        $this->blogCommentModel->set('deleted_at', NULL);
        $this->blogCommentModel->where('id', $post_id);
        $this->blogCommentModel->update();

        return redirect()->to(base_url('dashboard/deleted-blog-comments-list/'))->withInput()->with('success', lang('Dashboard.commentRestoredSuccessfully'));
    }


    //===========================================================//
    public function hard_delete_blog_comment()
    {
        // Check review exist
        $post_id = service('uri')->getSegment(3);
        $data['comment'] = $this->blogCommentModel->where('id', $post_id)->withDeleted()->first();
        if(is_null($data['comment']))
        {
            return redirect()->back()->with('error', lang('Dashboard.notFound'));
        }

        // hardDelete
        $this->blogCommentModel->delete($post_id);

        $db = db_connect();
        $builder = $db->table('blog_comments_tbl');
        $builder->delete(['id' => $post_id]);
        $db->close();

        return redirect()->to(base_url('dashboard/deleted-blog-comments-list/'))->withInput()->with('success', lang('Dashboard.dataDeletedSuccessfully'));
    }


    //===========================================================//
    public function edit_blog_comment()
    {
        $id = service('uri')->getSegment(3);
        $data['comment'] = $this->blogCommentModel->select('blog_comments_tbl.*')->where('blog_comments_tbl.id', $id)->withDeleted()->first();
        if(is_null($data['comment']))
            return redirect()->back()->with('error', lang('Dashboard.notFound'));

        $data['pageTitle'] = lang('Dashboard.editComment');
        $data['pageSubTitle'] = lang('Dashboard.editComment');
        return view('\App\Views\backend\blog\edit_blog_comment_view', $data);
    }


    //===========================================================//
    public function perform_edit_blog_comment()
    {
        $id = $this->request->getVar('id');
        // Validation the form
        $rules = [
            'id'        => 'required|integer',
            'comment'   => 'required|min_length[2]|max_length[1000]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url("dashboard/edit-blog-comment/$id"))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $data = [
            'id'        => $this->request->getVar('id'),
            'comment'   => $this->request->getVar('comment'),
        ];
        $this->blogCommentModel->save($data);

        return redirect()->to(base_url("dashboard/edit-blog-comment/$id"))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function pending_blog_comments_list()
    {
        $keyword = "";
        if(isset($_GET['keyword'])) $keyword = $_GET['keyword'];

        if(empty($keyword))
        {
            $data = [
                'blogComments' => $this->blogCommentModel
                    ->select('blog_comments_tbl.*')
                    ->where('blog_comments_tbl.approved', 2)
                    ->orderBy('blog_comments_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->blogCommentModel->pager
            ];

        }else{
            $data = [
                'blogComments' => $this->blogCommentModel
                    ->select('blog_comments_tbl.*')
                    ->like('blog_comments_tbl.author_name', $keyword)
                    ->orLike('blog_comments_tbl.author_email', $keyword)
                    ->orLike('blog_comments_tbl.author_ip', $keyword)
                    ->orLike('blog_comments_tbl.comment', $keyword)
                    ->where('blog_comments_tbl.approved', 2)
                    ->orderBy('blog_comments_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->blogCommentModel->pager
            ];
        }

        $data['pageTitle'] = lang('Dashboard.usersComments');
        $data['pageSubTitle'] = lang('Dashboard.pendingComments');
        return view('\App\Views\backend\blog\pending_blog_comments_view', $data);
    }


    //===========================================================//
    public function approved_blog_comments_list()
    {
        $keyword = "";
        if(isset($_GET['keyword'])) $keyword = $_GET['keyword'];

        if(empty($keyword))
        {
            $data = [
                'blogComments' => $this->blogCommentModel
                    ->select('blog_comments_tbl.*')
                    ->where('blog_comments_tbl.approved', 1)
                    ->orderBy('blog_comments_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->blogCommentModel->pager
            ];

        }else{
            $data = [
                'blogComments' => $this->blogCommentModel
                    ->select('blog_comments_tbl.*')
                    ->like('blog_comments_tbl.author_name', $keyword)
                    ->orLike('blog_comments_tbl.author_email', $keyword)
                    ->orLike('blog_comments_tbl.author_ip', $keyword)
                    ->orLike('blog_comments_tbl.comment', $keyword)
                    ->where('blog_comments_tbl.approved', 1)
                    ->orderBy('blog_comments_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->blogCommentModel->pager
            ];
        }

        $data['pageTitle'] = lang('Dashboard.usersComments');
        $data['pageSubTitle'] = lang('Dashboard.approvedComments');
        return view('\App\Views\backend\blog\approved_blog_comments_view', $data);
    }


    //===========================================================//
    public function unapproved_blog_comments_list()
    {
        $keyword = "";
        if(isset($_GET['keyword'])) $keyword = $_GET['keyword'];

        if(empty($keyword))
        {
            $data = [
                'blogComments' => $this->blogCommentModel
                    ->select('blog_comments_tbl.*')
                    ->where('blog_comments_tbl.approved', 0)
                    ->orderBy('blog_comments_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->blogCommentModel->pager
            ];

        }else{
            $data = [
                'blogComments' => $this->blogCommentModel
                    ->select('blog_comments_tbl.*')
                    ->like('blog_comments_tbl.author_name', $keyword)
                    ->orLike('blog_comments_tbl.author_email', $keyword)
                    ->orLike('blog_comments_tbl.author_ip', $keyword)
                    ->orLike('blog_comments_tbl.comment', $keyword)
                    ->where('blog_comments_tbl.approved', 0)
                    ->orderBy('blog_comments_tbl.id', 'DESC')
                    ->paginate(10),
                'pager' => $this->blogCommentModel->pager
            ];
        }

        $data['pageTitle'] = lang('Dashboard.usersComments');
        $data['pageSubTitle'] = lang('Dashboard.unapprovedComments');
        return view('\App\Views\backend\blog\unapproved_blog_comments_view', $data);
    }


    //===========================================================//
    public function approve_blog_comment()
    {
        // Check comment exist
        $comment_id = service('uri')->getSegment(3);
        $data['comment'] = $this->blogCommentModel->where('id', $comment_id)->withDeleted()->first();
        if(is_null($data['comment']))
        {
            return redirect()->back()->with('error', lang('Dashboard.notFound'));
        }

        $newData = [
            'id'           => $comment_id,
            'approved'     => 1,
        ];
        $this->blogCommentModel->save($newData);

        return redirect()->back()->with('success', lang('Dashboard.commentApprovedSuccessfully'));
    }


    //===========================================================//
    public function unapproved_blog_comment()
    {
        // Check comment exist
        $comment_id = service('uri')->getSegment(3);
        $data['comment'] = $this->blogCommentModel->where('id', $comment_id)->withDeleted()->first();
        if(is_null($data['comment']))
        {
            return redirect()->back()->with('error', lang('Dashboard.notFound'));
        }

        $newData = [
            'id'           => $comment_id,
            'approved'     => 0,
        ];
        $this->blogCommentModel->save($newData);

        return redirect()->back()->with('error', lang('Dashboard.commentUnapproved'));
    }
}
